// This module contains the actual encoding and decoding algorithms.
// Throws "RangeError" exceptions on characters or bytes out of range for the given encoding.
"use strict;"
var _this = this;

/* decoding error codes */
var NON_SHORTEST = 0xFFFFFFFC;
var TRAILING     = 0xFFFFFFFD;
var RANGE        = 0xFFFFFFFE;
var ILL_FORMED   = 0xFFFFFFFF;

/* mask[n] = 2**n - 1, ie. mask[n] = n bits on. e.g. mask[6] = %b111111 */
var mask = [0, 1, 3, 7, 15, 31, 63, 127, 255, 511, 1023];

/* ascii[n] = 'HH', where 0xHH = n, eg. ascii[254] = 'FE' */
var ascii = [ 
  '00', '01', '02', '03', '04', '05', '06', '07', '08', '09', '0A', '0B', '0C', '0D', '0E', '0F', '10', '11', '12', '13', '14', '15', '16', '17',
  '18', '19', '1A', '1B', '1C', '1D', '1E', '1F', '20', '21', '22', '23', '24', '25', '26', '27', '28', '29', '2A', '2B', '2C', '2D', '2E', '2F', '30',
  '31', '32', '33', '34', '35', '36', '37', '38', '39', '3A', '3B', '3C', '3D', '3E', '3F', '40', '41', '42', '43', '44', '45', '46', '47', '48', '49',
  '4A', '4B', '4C', '4D', '4E', '4F', '50', '51', '52', '53', '54', '55', '56', '57', '58', '59', '5A', '5B', '5C', '5D', '5E', '5F', '60', '61', '62',
  '63', '64', '65', '66', '67', '68', '69', '6A', '6B', '6C', '6D', '6E', '6F', '70', '71', '72', '73', '74', '75', '76', '77', '78', '79', '7A', '7B',
  '7C', '7D', '7E', '7F', '80', '81', '82', '83', '84', '85', '86', '87', '88', '89', '8A', '8B', '8C', '8D', '8E', '8F', '90', '91', '92', '93', '94',
  '95', '96', '97', '98', '99', '9A', '9B', '9C', '9D', '9E', '9F', 'A0', 'A1', 'A2', 'A3', 'A4', 'A5', 'A6', 'A7', 'A8', 'A9', 'AA', 'AB', 'AC', 'AD',
  'AE', 'AF', 'B0', 'B1', 'B2', 'B3', 'B4', 'B5', 'B6', 'B7', 'B8', 'B9', 'BA', 'BB', 'BC', 'BD', 'BE', 'BF', 'C0', 'C1', 'C2', 'C3', 'C4', 'C5', 'C6',
  'C7', 'C8', 'C9', 'CA', 'CB', 'CC', 'CD', 'CE', 'CF', 'D0', 'D1', 'D2', 'D3', 'D4', 'D5', 'D6', 'D7', 'D8', 'D9', 'DA', 'DB', 'DC', 'DD', 'DE', 'DF',
  'E0', 'E1', 'E2', 'E3', 'E4', 'E5', 'E6', 'E7', 'E8', 'E9', 'EA', 'EB', 'EC', 'ED', 'EE', 'EF', 'F0', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8',
  'F9', 'FA', 'FB', 'FC', 'FD', 'FE', 'FF' ];

/* vector of base 64 characters */
var base64chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=".split("");

/* vector of base 64 character codes */
var base64codes = [];
base64chars.forEach(function(char){
  base64codes.push(char.charCodeAt(0));
});

// The UTF8 algorithms.
exports.utf8 = {
    encode : function(chars) {
      var bytes = [];
      chars.forEach(function(char){
        if(char >= 0 && char <= 0x7f){
          bytes.push(char);
        }else if(char <= 0x7ff){
          bytes.push(0xc0 + ((char >> 6) & mask[5]));
          bytes.push(0x80 + (char & mask[6]));
        }else if(char < 0xD800 || (char > 0xDFFF && char <= 0xffff)){
          bytes.push(0xe0 + ((char >> 12) & mask[4]));
          bytes.push(0x80 + ((char >> 6) & mask[6]));
          bytes.push(0x80 + (char & mask[6]));
        }else if(char >= 0x10000 && char <= 0x10ffff){
          var u = (char >> 16) & mask[5];
          bytes.push(0xf0 + (u >> 2));
          bytes.push(0x80 + ((u & mask[2]) << 4) + ((char >> 12) & mask[4]));
          bytes.push(0x80 + ((char >> 6) & mask[6]));
          bytes.push(0x80 + (char & mask[6]));
        }else{
          throw new RangeError("utf8.encode: character out of range: char: " + char);
        }
      });
      return Buffer.from(bytes);
    },
  decode : function(buf, bom) {
    /* bytes functions return error for non-shortest forms & values out of range */
    function bytes2(b1, b2){
      /*U+0080..U+07FF */
      /* 00000000 00000yyy yyxxxxxx | 110yyyyy 10xxxxxx */
      if((b2 & 0xC0) !== 0x80){
        return TRAILING;
      }
      var x = ((b1 & mask[5]) << 6) + (b2 & mask[6]);
      if(x < 0x80){
        return NON_SHORTEST;
      }
      return x;
    }
    function bytes3(b1, b2, b3){
      /*U+0800..U+FFFF */
      /* 00000000 zzzzyyyy yyxxxxxx | 1110zzzz 10yyyyyy 10xxxxxx */
      if(((b3 & 0xC0) !== 0x80) || ((b2 & 0xC0) !== 0x80)){
        return TRAILING;
      }
      var x = ((b1 & mask[4]) << 12) + ((b2 & mask[6]) << 6) + (b3 & mask[6]);
      if(x < 0X800){
        return NON_SHORTEST;
      }
      if((x >= 0xD800) && (x <= 0xDFFF)){
        return RANGE;
      }
      return x;
    }
    function bytes4(b1, b2, b3, b4){
      /*U+10000..U+10FFFF */
      /* 000uuuuu zzzzyyyy yyxxxxxx | 11110uuu 10uuzzzz 10yyyyyy 10xxxxxx */
      if(((b4 & 0xC0) !== 0x80) || ((b3 & 0xC0) !== 0x80) || ((b2 & 0xC0) !== 0x80)){
        return TRAILING;
      }
      var x = ((((b1 & mask[3]) << 2)
          + ((b2 >> 4) & mask[2])) << 16)
          + ((b2 & mask[4]) << 12)
          + ((b3 & mask[6]) << 6)
          + (b4 & mask[6]);
      if(x < 0x10000){
        return NON_SHORTEST;
      }
      if(x > 0x10FFFF){
        return RANGE;
      }
      return x;
    }
    var c, b1, i1, i2, i3, inc;
    var len = buf.length;
    var i = bom ? 3 : 0;
    var chars = [];
    while(i < len){
      b1 = buf[i];
      c = ILL_FORMED;
      while(true){
        if(b1 >=0 && b1 <=0x7f){
          /*U+0000..U+007F 00..7F*/
          c = b1;
          inc = 1;
          break;
        }
        i1 = i + 1;
        if((i1 < len) && (b1 >= 0xc2 && b1 <= 0xdf)){
          /*U+0080..U+07FF C2..DF 80..BF*/
          c = bytes2(b1, buf[i1]);
          inc = 2;
          break;
        }
        i2 = i + 2
        if((i2 < len) && (b1 >= 0xe0 && b1 <= 0xef)){
          /*U+0800..U+FFFF */
          c = bytes3(b1, buf[i1], buf[i2]);
          inc = 3;
          break;
        }
        i3 = i + 3
        if((i3 < len) && (b1 >= 0xf0 && b1 <= 0xf4)){
          /*U+10000..U+10FFFF */
          c = bytes4(b1, buf[i1], buf[i2], buf[i3]);
          inc = 4;
          break;
        }
        /* if we fall through to here, it is an ill-formed sequence*/
        break;
      }
      if(c > 0x10FFFF){
        var at = "byte["+i+"]";
        if(c === ILL_FORMED){
          throw new RangeError("utf8.decode: ill-formed UTF8 byte sequence found at: "+at);
        }
        if(c === TRAILING){
          throw new RangeError("utf8.decode: illegal trailing byte found at: "+at);
        }
        if(c === RANGE){
          throw new RangeError("utf8.decode: code point out of range found at: "+at);
        }
        if(c === NON_SHORTEST){
          throw new RangeError("utf8.decode: non-shortest form found at: "+at);
        }
        throw new RangeError("utf8.decode: unrecognized error found at: "+at);
      }
      chars.push(c);
      i += inc;
    }
    return chars;
  },
}

//The UTF16BE algorithms.
exports.utf16be = {
  encode : function(chars) {
    var bytes = [];
    var char, h, l;
    for(var i = 0; i < chars.length; i += 1){
      char = chars[i];
      if( ((char >= 0) && (char <= 0xD7FF)) || ((char >= 0xE000) && (char <= 0xFFFF)) ){
        bytes.push((char >> 8) & mask[8]);
        bytes.push(char & mask[8]);
      }else if(char >= 0x10000 && char <= 0x10FFFF){
        l = char - 0x10000;
        h = 0xD800 + (l >> 10);
        l = 0xDC00 + (l & mask[10]);
        bytes.push((h >> 8) & mask[8]);
        bytes.push(h & mask[8]);
        bytes.push((l >> 8) & mask[8]);
        bytes.push(l & mask[8]);
      }else{
        throw new RangeError("utf16be.encode: UTF16BE value out of range: char["+i+"]: "+char);
      }
    }
    return Buffer.from(bytes);
  },
  decode : function(buf, bom) {
    /* assumes caller has insured that buf is a Buffer of bytes */
    if(buf.length % 2 > 0){
      throw new RangeError("utf16be.decode: data length must be even multiple of 2: length: "+buf.length);
    }
    var chars = [];
    var len = buf.length;
    var i = bom ? 2 : 0;
    var j = 0;
    var c, inc, i1, i3, high, low;
    while(i < len){
      while(true){
        i1 = i + 1;
        if(i1 < len){
          high = (buf[i] << 8) + buf[i1];
          if((high < 0xD800) || (high > 0xDFFF)){
            c = high;
            inc = 2;
            break;
          }
          i3 = i + 3;
          if(i3 < len){
            low = (buf[i + 2] << 8) + buf[i3];
            if((high <= 0xDBFF) && (low >= 0xDC00) && (low <= 0xDFFF)){
              c = 0x10000 + ((high - 0xD800) << 10) + (low - 0xDC00);
              inc = 4;
              break;
            }
          }
        }
        /* if we fall through to here, it is an ill-formed sequence*/
        throw new RangeError("utf16be.decode: ill-formed UTF16BE byte sequence found: byte["+i+"]");
      }
      chars[j++] = c;
      i += inc;
    }
    return chars;
  },
}

//The UTF16LE algorithms.
exports.utf16le = {
  encode : function(chars) {
    var bytes = [];
    var char, h, l;
    for(var i = 0; i < chars.length; i += 1){
      char = chars[i];
      if( ((char >= 0) && (char <= 0xD7FF)) || ((char >= 0xE000) && (char <= 0xFFFF)) ){
        bytes.push(char & mask[8]);
        bytes.push((char >> 8) & mask[8]);
      }else if(char >= 0x10000 && char <= 0x10FFFF){
        l = char - 0x10000;
        h = 0xD800 + (l >> 10);
        l = 0xDC00 + (l & mask[10]);
        bytes.push(h & mask[8]);
        bytes.push((h >> 8) & mask[8]);
        bytes.push(l & mask[8]);
        bytes.push((l >> 8) & mask[8]);
      }else{
        throw new RangeError("utf16le.encode: UTF16LE value out of range: char["+i+"]: "+char);
      }
    }
    return Buffer.from(bytes);
  },
  decode : function(buf, bom) {
    /* assumes caller has insured that buf is a Buffer of bytes */
    if(buf.length % 2 > 0){
      throw new RangeError("utf16le.decode: data length must be even multiple of 2: length: "+buf.length);
    }
    var chars = [];
    var len = buf.length;
    var i = bom ? 2 : 0;
    var j = 0;
    var c, inc, i1, i3, high, low;
    while(i < len){
      while(true){
        i1 = i + 1;
        if(i1 < len){
          high = (buf[i1] << 8) + buf[i];
          if((high < 0xD800) || (high > 0xDFFF)){
            c = high;
            inc = 2;
            break;
          }
          i3 = i + 3;
          if(i3 < len){
            low = (buf[i3] << 8) + buf[i + 2];
            if((high <= 0xDBFF) && (low >= 0xDC00) && (low <= 0xDFFF)){
              c = 0x10000 + ((high - 0xD800) << 10) + (low - 0xDC00);
              inc = 4;
              break;
            }
          }
        }
        /* if we fall through to here, it is an ill-formed sequence*/
        throw new RangeError("utf16le.decode: ill-formed UTF16LE byte sequence found: byte["+i+"]");
      }
      chars[j++] = c;
      i += inc;
    }
    return chars;
  },
}

//The UTF32BE algorithms.
exports.utf32be = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length * 4);
    var i = 0;
    chars.forEach(function(char){
      if(((char >= 0xD800) && (char <= 0xDFFF)) || (char > 0x10FFFF)){
        throw new RangeError("utf32be.encode: UTF32BE character code out of range: char["+i/4+"]: " + char);
      }
      buf[i++] = (char >> 24) & mask[8];
      buf[i++] = (char >> 16) & mask[8];
      buf[i++] = (char >> 8) & mask[8];
      buf[i++] = char & mask[8];
    });
    return buf;
  },
  decode : function(buf, bom) {
    /* caller to insure buf is a Buffer of bytes */
    if(buf.length % 4 > 0 ){
      throw new RangeError("utf32be.decode: UTF32BE byte length must be even multiple of 4: length: " + buf.length);
    }
    var chars = [];
    var i = bom ? 4 : 0;
    for(; i < buf.length; i += 4){
      var char = (buf[i]<<24) + (buf[i+1]<<16) + (buf[i+2]<<8) + buf[i+3];
      if(((char >= 0xD800) && (char <= 0xDFFF)) || (char > 0x10FFFF)){
        throw new RangeError("utf32be.decode: UTF32BE character code out of range: char["+i/4+"]: " + char);
      }
      chars.push(char);
    }
    return chars;
  }
}

//The UTF32LE algorithms.
exports.utf32le = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length * 4);
    var i = 0;
    chars.forEach(function(char){
      if(((char >= 0xD800) && (char <= 0xDFFF)) || (char > 0x10FFFF)){
        throw new RangeError("utf32le.encode: UTF32LE character code out of range: char["+i/4+"]: " + char);
      }
      buf[i++] = char & mask[8];
      buf[i++] = (char >> 8) & mask[8];
      buf[i++] = (char >> 16) & mask[8];
      buf[i++] = (char >> 24) & mask[8];
    });
    return buf;
  },
  decode : function(buf, bom) {
    /* caller to insure buf is a Buffer of bytes */
    if(buf.length % 4 > 0 ){
      throw new RangeError("utf32be.decode: UTF32LE byte length must be even multiple of 4: length: " + buf.length);
    }
    var chars = [];
    var i = bom ? 4 : 0;
    for(; i < buf.length; i += 4){
      var char = (buf[i+3]<<24) + (buf[i+2]<<16) + (buf[i+1]<<8) + buf[i];
      if(((char >= 0xD800) && (char <= 0xDFFF)) || (char > 0x10FFFF)){
        throw new RangeError("utf32le.encode: UTF32LE character code out of range: char["+i/4+"]: " + char);
      }
      chars.push(char);
    }
    return chars;
  }
}

//The UINT7 algorithms. ASCII or 7-bit unsigned integers.
exports.uint7 = {
    encode : function(chars) {
      var buf = Buffer.alloc(chars.length);
      for(var i = 0; i < chars.length; i += 1){
        if(chars[i] > 0x7f){
          throw new RangeError("uint7.encode: UINT7 character code out of range: char["+i+"]: " + chars[i]);
        }
        buf[i] = chars[i];
      }
      return buf;
    },
    decode : function(buf) {
      var chars = [];
      for(var i = 0; i < buf.length; i += 1){
        if(buf[i] > 0x7f){
          throw new RangeError("uint7.decode: UINT7 character code out of range: byte["+i+"]: " + buf[i]);
        }
        chars[i] = buf[i];
      }
      return chars;
    }
  }

//The UINT8 algorithms. BINARY, Latin 1 or 8-bit unsigned integers.
exports.uint8 = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length);
    for(var i = 0; i < chars.length; i += 1){
      if(chars[i] > 0xff){
        throw new RangeError("uint8.encode: UINT8 character code out of range: char["+i+"]: " + chars[i]);
      }
      buf[i] = chars[i];
    }
    return buf;
  },
  decode : function(buf) {
    var chars = [];
    for(var i = 0; i < buf.length; i += 1){
      chars[i] = buf[i];
    }
    return chars;
  }
}

//The UINT16BE algorithms. Big-endian 16-bit unsigned integers.
exports.uint16be = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length * 2);
    var i = 0;
    chars.forEach(function(char){
      if(char > 0xffff){
        throw new RangeError("uint16be.encode: UINT16BE character code out of range: char["+(i/2)+"]: " + char);
      }
      buf[i++] = (char >> 8) & mask[8];
      buf[i++] = char & mask[8];
    });
    return buf;
  },
  decode : function(buf) {
    if(buf.length % 2 > 0 ){
      throw new RangeError("uint16be.decode: UINT16BE byte length must be even multiple of 2: length: " + buf.length);
    }
    var chars = [];
    for(var i = 0; i < buf.length; i += 2){
      chars.push((buf[i]<<8) + buf[i+1]);
    }
    return chars;
  }
}

//The UINT16LE algorithms. Little-endian 16-bit unsigned integers.
exports.uint16le = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length * 2);
    var i = 0;
    chars.forEach(function(char){
      if(char > 0xffff){
        throw new RangeError("uint16le.encode: UINT16LE character code out of range: char["+(i/2)+"]: " + char);
      }
      buf[i++] = char & mask[8];
      buf[i++] = (char >> 8) & mask[8];
    });
    return buf;
  },
  decode : function(buf) {
    if(buf.length % 2 > 0 ){
      throw new RangeError("uint16le.decode: UINT16LE byte length must be even multiple of 2: length: " + buf.length);
    }
    var chars = [];
    for(var i = 0; i < buf.length; i += 2){
      chars.push((buf[i+1]<<8) + buf[i]);
    }
    return chars;
  }
}

//The UINT32BE algorithms. Big-endian 32-bit unsigned integers.
exports.uint32be = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length * 4);
    var i = 0;
    chars.forEach(function(char){
      buf[i++] = (char >> 24) & mask[8];
      buf[i++] = (char >> 16) & mask[8];
      buf[i++] = (char >> 8) & mask[8];
      buf[i++] = char & mask[8];
    });
    return buf;
  },
  decode : function(buf) {
    if(buf.length % 4 > 0 ){
      throw new RangeError("uint32be.decode: UINT32BE byte length must be even multiple of 4: length: " + buf.length);
    }
    var chars = [];
    for(var i = 0; i < buf.length; i += 4){
      chars.push((buf[i]<<24) + (buf[i+1]<<16) + (buf[i+2]<<8) + buf[i+3]);
    }
    return chars;
  }
}

//The UINT32LE algorithms. Little-endian 32-bit unsigned integers.
exports.uint32le = {
  encode : function(chars) {
    var buf = Buffer.alloc(chars.length * 4);
    var i = 0;
    chars.forEach(function(char){
      buf[i++] = char & mask[8];
      buf[i++] = (char >> 8) & mask[8];
      buf[i++] = (char >> 16) & mask[8];
      buf[i++] = (char >> 24) & mask[8];
    });
    return buf;
  },
  decode : function(buf) {
    /* caller to insure buf is a Buffer of bytes */
    if(buf.length % 4 > 0 ){
      throw new RangeError("uint32le.decode: UINT32LE byte length must be even multiple of 4: length: " + buf.length);
    }
    var chars = [];
    for(var i = 0; i < buf.length; i += 4){
      chars.push((buf[i+3]<<24) + (buf[i+2]<<16) + (buf[i+1]<<8) + buf[i]);
    }
    return chars;
  }
}

// The STRING algorithms. Converts JavaScript strings to Array of 32-bit integers and vice versa. 
// Uses the node.js Buffer's native "utf16le" capabilites.
exports.string = {
  encode: function(chars){
    return _this.utf16le.encode(chars).toString("utf16le");
  },
  decode: function(str){
    return _this.utf16le.decode(Buffer.from(str, "utf16le"), 0);
  }
}

//The ESCAPED algorithms. 
// Note that ESCAPED format contains only ASCII characters.
// The characters are always in the form of a Buffer of bytes.
exports.escaped = {
// Encodes an Array of 32-bit integers into ESCAPED format.
  encode : function(chars) {
    var bytes = [];
    for (var i = 0; i < chars.length; i += 1) {
      var char = chars[i];
      if (char === 96) {
        bytes.push(char);
        bytes.push(char);
      } else if (char === 10) {
        bytes.push(char);
      } else if (char >= 32 && char <= 126) {
        bytes.push(char);
      } else{
        var str = "";
        if (char >= 0 && char <= 31) {
          str += "`x" + ascii[char];
        } else if (char >= 127 && char <= 255) {
          str += "`x" + ascii[char];
        } else if (char >= 0x100 && char <= 0xffff) {
          str += "`u" + ascii[(char >> 8) & mask[8]] + ascii[char & mask[8]];
        } else if (char >= 0x10000 && char <= 0xffffffff) {
          str += "`u{";
          var digit = (char >> 24) & mask[8];
          if (digit > 0) {
            str += ascii[digit];
          }
          str += ascii[(char >> 16) & mask[8]] + ascii[(char >> 8) & mask[8]] + ascii[char & mask[8]] + "}";
        } else {
          throw new Error("escape.encode(char): char > 0xffffffff not allowed");
        }
        var buf = Buffer.from(str);
        buf.forEach(function(b){
          bytes.push(b);
        });
      }
    }
    return Buffer.from(bytes);
  },
  // Decodes ESCAPED format from a Buffer of bytes to an Array of 32-bit integers.
  decode : function(buf) {
    function isHex(hex){
      if((hex >= 48 && hex <= 57) || (hex >= 65 && hex <= 70) || (hex >= 97 && hex <= 102)){
        return true;
      }
      return false;
    }
    function getx(i, len, buf){
      var ret = {char: null, nexti: i + 2, error: true};
      if(i + 1 < len ){
        if(isHex(buf[i]) && isHex(buf[i+1])){
          var str = String.fromCodePoint(buf[i], buf[i+1]);
          ret.char = parseInt(str, 16);
          if(!isNaN(ret.char)){
            ret.error = false;
          }
        }
      }
      return ret;
    }
    function getu(i, len, buf){
      var ret = {char: null, nexti: i + 4, error: true};
      if(i + 3 < len ){
        if(isHex(buf[i]) && isHex(buf[i+1]) && isHex(buf[i+2]) && isHex(buf[i+3])){
          var str = String.fromCodePoint(buf[i], buf[i+1], buf[i+2], buf[i+3]);
          ret.char = parseInt(str, 16);
          if(!isNaN(ret.char)){
            ret.error = false
          }
        }
      }
      return ret;
    }
    function getU(i, len, buf){
      var ret = {char: null, nexti: i + 4, error: true};
      var str = "";
      while(i < len && isHex(buf[i])){
        str += String.fromCodePoint(buf[i]);
        i += 1;
      }
      ret.char = parseInt(str, 16);
      if(buf[i] === 125 && !isNaN(ret.char)){
        ret.error = false
      }
      ret.nexti = i + 1;
      return ret;
    }
    var chars = [];
    var len = buf.length;
    var i1, ret, error;
    var i = 0;
    while(i < len){
      while(true){
        error = true;
        if(buf[i] !== 96){
          /* unescaped character */
          chars.push(buf[i]);
          i += 1;
          error = false;
          break;
        }
        i1 = i + 1;
        if(i1 >= len){
          break;
        }
        if(buf[i1] === 96){
          /* escaped grave accent */
          chars.push(96);
          i += 2;
          error = false;
          break;
        }
        if(buf[i1] === 120){
          ret = getx(i1+1, len, buf);
          if(ret.error){
            break;
          }
          /* escaped hex */
          chars.push(ret.char);
          i = ret.nexti;
          error = false;
          break;
        }
        if(buf[i1] === 117){
          if(buf[i1+1] === 123){
            ret = getU(i1 + 2, len, buf);
            if(ret.error){
              break;
            }
            /* escaped utf-32 */
            chars.push(ret.char);
            i = ret.nexti;
            error = false;
            break;
          }
          ret = getu(i1 + 1, len, buf);
          if(ret.error){
            break;
          }
          /* escaped utf-16 */
          chars.push(ret.char);
          i = ret.nexti;
          error = false;
          break;
        }
        break;
      }
      if(error){
        throw new Error("escaped.decode: ill-formed escape sequence at buf["+i+"]");
      }
    }
    return chars;
  }
}

// The line end conversion algorigthms.
var CR = 13;
var LF = 10;
exports.lineEnds = {
    crlf: function(chars){
      var lfchars = [];
      var i = 0;
      while(i < chars.length){
        switch(chars[i]){
        case CR:
          if((i + 1) < chars.length && chars[i + 1] === LF){
            i += 2;
          }else{
            i += 1
          }
          lfchars.push(CR);
          lfchars.push(LF);
          break;
        case LF:
          lfchars.push(CR);
          lfchars.push(LF);
          i += 1;
          break;
        default:
          lfchars.push(chars[i]);
          i += 1;
          break;
        }
      }
      if(lfchars.length > 0 && lfchars[lfchars.length - 1] !== LF){
        lfchars.push(CR);
        lfchars.push(LF);
      }
      return lfchars;
    },
    lf: function(chars){
      var lfchars = [];
      var i = 0;
      while(i < chars.length){
        switch(chars[i]){
        case CR:
          if((i + 1) < chars.length && chars[i + 1] === LF){
            i += 2;
          }else{
            i += 1
          }
          lfchars.push(LF);
          break;
        case LF:
          lfchars.push(LF);
          i += 1;
          break;
        default:
          lfchars.push(chars[i]);
          i += 1;
          break;
        }
      }
      if(lfchars.length > 0 && lfchars[lfchars.length - 1] !== LF){
        lfchars.push(LF);
      }
      return lfchars;
    }
}

// The base 64 algorithms.
exports.base64 = {
  encode : function(buf) {
    if(buf.length === 0){
      return Buffer.alloc(0);
    }
    var i, j, n;
    var tail = buf.length % 3;
    tail = (tail > 0) ? 3-tail : 0;
    var units = (buf.length + tail)/3;
    var base64 = Buffer.alloc(units * 4);
    if(tail > 0){
      units -= 1;
    }
    i = 0;
    j = 0;
    for(var u = 0; u < units; u += 1){
      n = buf[i++] << 16;
      n += buf[i++] << 8;
      n += buf[i++];
      base64[j++] = base64codes[(n >> 18) & mask[6]];
      base64[j++] = base64codes[(n >> 12) & mask[6]];
      base64[j++] = base64codes[(n >> 6) & mask[6]];
      base64[j++] = base64codes[n & mask[6]];
    }
    if (tail === 0) {
      return base64;
    }
    if (tail === 1) {
      n = buf[i++] << 16;
      n += buf[i] << 8;
      base64[j++] = base64codes[(n >> 18) & mask[6]];
      base64[j++] = base64codes[(n >> 12) & mask[6]];
      base64[j++] = base64codes[(n >> 6) & mask[6]];
      base64[j] = base64codes[64];
      return base64;
    }
    if (tail === 2) {
      n = buf[i] << 16;
      base64[j++] = base64codes[(n >> 18) & mask[6]];
      base64[j++] = base64codes[(n >> 12) & mask[6]];
      base64[j++] = base64codes[64];
      base64[j] = base64codes[64];
      return base64;
    }
  },
  decode: function(codes){
    /* remove white space and ctrl characters, validate & translate characters */
    function validate(buf){
      var chars = [];
      var tail = 0;
      for (var i = 0; i < buf.length; i += 1) {
        var char = buf[i];
        while(true){
          if (char === 32 || char === 9 || char === 10 || char === 13) {
            break;
          }
          if (char >= 65 && char <= 90) {
            chars.push(char - 65);
            break;
          }
          if (char >= 97 && char <= 122) {
            chars.push(char - 71);
            break;
          }
          if (char >= 48 && char <= 57) {
            chars.push(char + 4);
            break;
          }
          if (char === 43) {
            chars.push(62);
            break;
          }
          if (char === 47) {
            chars.push(63);
            break;
          }
          if (char === 61) {
            chars.push(64);
            tail += 1;
            break;
          }
          /* invalid character */
          throw new RangeError("base64.decode: invalid character buf[" + i + "]: " + char);
        }
      }
      /* validate length */
      if (chars.length % 4 > 0) {
        throw new RangeError("base64.decode: string length not integral multiple of 4: " + chars.length);
      }
      /* validate tail */
      switch(tail){
      case 0:
        break;
      case 1:
        if (chars[chars.length - 1] !== 64) {
          throw new RangeError("base64.decode: one tail character found: not last character");
        }
        break;
      case 2:
        if ((chars[chars.length - 1] !== 64) || (chars[chars.length - 2] !== 64)) {
          throw new RangeError("base64.decode: two tail characters found: not last characters");
        }
        break;
      default:
        throw new RangeError("base64.decode: more than two tail characters found: " + tail);
      }
      return {tail: tail, buf: Buffer.from(chars)}
    }
    
    if(codes.length === 0){
      return Buffer.alloc(0);
    }
    var val = validate(codes);
    var tail = val.tail;
    var base64 = val.buf;
    var i, j, n;
    var units = base64.length / 4;
    var buf = Buffer.alloc((units*3) - tail);
    if(tail > 0){
      units -= 1;
    }
    j = 0;
    i = 0;
    for(var u = 0; u < units; u += 1) {
      n = base64[i++] << 18;
      n += base64[i++] << 12;
      n += base64[i++] << 6;
      n += base64[i++];
      buf[j++] = (n >> 16) & mask[8];
      buf[j++] = (n >> 8) & mask[8];
      buf[j++] = n & mask[8];
    }
    if (tail === 1) {
      n = base64[i++] << 18;
      n += base64[i++] << 12;
      n += base64[i] << 6;
      buf[j++] = (n >> 16) & mask[8];
      buf[j] = (n >> 8) & mask[8];
    }
    if (tail === 2) {
      n = base64[i++] << 18;
      n += base64[i++] << 12;
      buf[j] = (n >> 16) & mask[8];
    }
    return buf;
    
  },
  // Converts a base 64 Buffer of bytes to a JavaScript string with line breaks.
  toString: function(buf){
    if(buf.length % 4 > 0){
      throw new RangeError("base64.toString: input buffer length not multiple of 4: " + buf.length);
    }
    var str = "";
    var lineLen = 0;
    function buildLine(c1, c2, c3, c4) {
      switch (lineLen) {
      case 76:
        str += "\r\n" + c1 + c2 + c3 + c4;
        lineLen = 4;
        break;
      case 75:
        str += c1 + "\r\n" + c2 + c3 + c4;
        lineLen = 3;
        break;
      case 74:
        str += c1 + c2 + "\r\n" + c3 + c4;
        lineLen = 2;
        break;
      case 73:
        str += c1 + c2 + c3 + "\r\n" + c4;
        lineLen = 1;
        break;
      default:
        str += c1 + c2 + c3 + c4;
        lineLen += 4;
        break;
      }
    }
    function validate(c){
      if(c >= 65 && c <= 90){
        return true
      }
      if(c >= 97 && c <= 122){
        return true
      }
      if(c >= 48 && c <= 57){
        return true
      }
      if(c === 43){
        return true
      }
      if(c === 47){
        return true
      }
      if(c === 61){
        return true
      }
      return false;
    }
    for(var i = 0; i < buf.length; i += 4){
      for(var j = i; j < i+4; j +=1){
        if(!validate(buf[j])){
          throw new RangeError("base64.toString: buf["+j+"]: "+buf[j]+" : not valid base64 character code");
        }
      }
      buildLine(String.fromCharCode(buf[i]), String.fromCharCode(buf[i+1]), String.fromCharCode(buf[i+2]), String.fromCharCode(buf[i+3]));
      
    }
    return str;
  },
}
